<?php
/* * **************************************************************************************
 *
 *  Copyright notice
 *
 *  (c) 2016 Harald Holzmann <harald@varioous.at>, varioous OG
 *
 *  Eyepin For Wordpress is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  any later version.
 *   
 *  Eyepin For Wordpress is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Eyepin For Wordpress. If not, see http://www.gnu.org/copyleft/gpl.html.
 *
 *  This copyright notice MUST APPEAR in all copies of the script!
 * ************************************************************************************** */

defined('ABSPATH') or die('No script kiddies please!');

if (!class_exists('WP_List_Table')) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

/**
 * Class for list table all forms
 * see https://www.sitepoint.com/using-wp_list_table-to-create-wordpress-admin-tables/ for more information
 */
class VA_EY4WP_Admin_Forms extends WP_List_Table {

	const TABLE_NAME = 'va_ey4wp_forms';
	const TABLE_NAME_EXPORTED = 'va_ey4wp_eyepin_exported';

	/** Class constructor */
	public function __construct() {

		parent::__construct([
			'singular' => esc_html__('Formular', 'eyepin'),
			'plural' => esc_html__('Formulare', 'eyepin'),
			'ajax' => false
		]);
	}

    /**
     * @param int $per_page
     * @param int $page_number
     *
     * @return mixed
     */
    public static function get_forms($per_page = 5, $page_number = 1) {
        global $wpdb;
        $table_name = "{$wpdb->prefix}va_ey4wp_forms";

        if (!empty($_REQUEST['orderby']) && wp_verify_nonce(!empty($_POST['form_valid_nonce']) ? map_deep( wp_unslash( $_POST['form_valid_nonce'] ) , 'sanitize_text_field' ) : '' , 'eyepin_admin_forms_data_nonce')) {
            $result = $wpdb->get_results($wpdb->prepare("SELECT * FROM %i ORDER BY %s ASC LIMIT %d OFFSET %d", $table_name, sanitize_text_field(wp_unslash( $_REQUEST['orderby'] )), $per_page, ( $page_number - 1 ) * $per_page ), 'ARRAY_A'); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        } else {
            $result = $wpdb->get_results($wpdb->prepare("SELECT * FROM %i LIMIT %d OFFSET %d", $table_name, $per_page, ( $page_number - 1 ) * $per_page ), 'ARRAY_A'); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        }

        return $result;
    }

	/**
	 * Delete a form record.
	 *
	 * @param int $id form ID
	 */
	public static function delete_form($id) {
		global $wpdb;
		$wpdb->delete("{$wpdb->prefix}" . self::TABLE_NAME, [ 'id' => $id], [ '%d']); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
	}

    /**
     * Returns the count of records in the database.
     *
     * @return null|string
     */
    public static function record_count() {
        global $wpdb;
        $table_name = "{$wpdb->prefix}va_ey4wp_forms";
        return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM %i", $table_name)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    }

	/**
	 * Display when no records available
	 */
	public function no_items() {
        esc_html_e('Keine Formulare verfügbar.', 'eyepin');
	}

    /**
     * Method for name column
     *
     * @param array $item an array of DB data
     *
     * @return string
     */
    function column_name($item) {
        // create a nonce
        $delete_nonce = wp_create_nonce('sp_delete_form');
        $currentScreen = get_current_screen();
        $currentPage = $currentScreen->parent_base . '-forms';
        $delete_link = sprintf(
            '<a href="?page=%s&action=%s&form=%s&_wpnonce=%s">' . esc_html__('Löschen', 'eyepin') . '</a>',
            $currentPage,
            'delete',
            absint($item['id']),
            $delete_nonce
        );
        $title = '<strong><a href="?page=eyepin-for-wp-forms&action=edit&form_id=' . $item['id'] . '">' . $item['name'] . '</a></strong>';
        $actions = [
            'edit' => '<a href="?page=eyepin-for-wp-forms&action=edit&form_id=' . $item['id'] . '">' . esc_html__('Bearbeiten', 'eyepin') . '</a>',
            'delete' => $delete_link
        ];
        return $title . $this->row_actions($actions);
    }


	/**
	 * Render a column when no column specific method exists.
	 *
	 * @param array $item
	 * @param string $column_name
	 *
	 * @return mixed
	 */
	public function column_default($item, $column_name) {
		switch ($column_name) {
			case 'name':
			case 'shortcode':
				$output = '[eyepin_for_wp id="' . $item['id'] . '" form="' . $item['name'] . '"]';
				return '<span class="shortcode"><input type="text"'
						. ' onfocus="this.select();" readonly="readonly"'
						. ' value="' . esc_attr($output) . '"'
						. ' class="large-text code" /></span>';
			case 'opt_in':
				return $item['opt_in'];
			default:
//				return print_r($item, true);
		}
	}

	/**
	 * Render the bulk edit checkbox
	 *
	 * @param array $item
	 *
	 * @return string
	 */
	function column_cb($item) {
		return sprintf(
				'<input type="checkbox" name="bulk-delete[]" value="%s" />', $item['id']
		);
	}

	/**
	 *  Associative array of columns
	 *
	 * @return array
	 */
	function get_columns() {
		$columns = [
			'cb' => '<input type="checkbox" />',
			'name' => esc_html__('Name', 'eyepin'),
			'shortcode' => esc_html__('Shortcode', 'eyepin'),
			'opt_in' => esc_html__('Opt-In durch Eyepin', 'eyepin')
		];

		return $columns;
	}

	/**
	 * Columns to make sortable.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {
		$sortable_columns = array(
			'name' => array('name', true)
		);

		return $sortable_columns;
	}

	/**
	 * Returns an associative array containing the bulk action
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		$actions = [
			'bulk-delete' => 'Delete'
		];

		return $actions;
	}

    /**
     * process bulk action
     */
    public function process_bulk_action() {
        //Detect when a bulk action is being triggered...
        if ('delete' === $this->current_action()) {
            // In our file that handles the request, verify the nonce.
            $nonce = !empty($_REQUEST['_wpnonce']) ? sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])) : '';
            if (!wp_verify_nonce($nonce, 'sp_delete_form')) {
                die('Go get a life script kiddies');
            } else {
                self::delete_form(absint($_GET['form'] ?? 0));
            }
        }

        // If the delete bulk action is triggered
        if (( isset($_POST['action']) && $_POST['action'] == 'bulk-delete' ) || ( isset($_POST['action2']) && $_POST['action2'] == 'bulk-delete' )
        ) {
            $delete_ids = !empty($_POST['bulk-delete']) ? map_deep( wp_unslash( $_POST['bulk-delete'] ) , 'sanitize_text_field' ) : [];

            // loop over the array of record IDs and delete them
            foreach ($delete_ids as $id) {
                self::delete_form($id);
            }
        }
    }

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {
		$this->_column_headers = $this->get_column_info();

		/** Process bulk action */
		$this->process_bulk_action();

		$per_page = $this->get_items_per_page('forms_per_page', 5);
		$current_page = $this->get_pagenum();
		$total_items = self::record_count();

		$columns = $this->get_columns();
		$sortable = $this->get_sortable_columns();
		$this->_column_headers = array($columns, array(), $sortable);

		$this->set_pagination_args(array(
			'total_items' => $total_items,
			'per_page' => $per_page
		));

		$this->items = self::get_forms($per_page, $current_page);
	}

	/**
	 * diplay forms table
	 */
	public function render() {
        $nonce = wp_create_nonce('eyepin_admin_forms_data_nonce');
		?>
		<div class="wrap">
			<h1 class="page-title"><?php esc_html_e('eyepin', 'eyepin') ?> › <?php esc_html_e('Formulare', 'eyepin') ?> <a href="<?php echo esc_url(site_url()); ?>/wp-admin/admin.php?page=eyepin-for-wp-forms&action=new" class="add-new-h2"><?php esc_html_e('Neu hinzufügen', 'eyepin') ?></a></h1>
			<div id="poststuff">
				<div id="post-body-content">
					<div class="meta-box-sortables ui-sortable">
						<form method="post">
                            <input type="hidden" name="form_valid_nonce" value="<?php echo esc_attr($nonce); ?>">
							<?php
							$this->prepare_items();
							$this->display();
							?>
						</form>
					</div>
				</div>
				<br class="clear">
			</div>
		</div>
		<?php
	}

}
