<?php


/* * **************************************************************************************

 *

 *  Copyright notice

 *

 *  (c) 2016 Harald Holzmann <harald@varioous.at>, varioous OG

 *

 *  Eyepin For Wordpress is free software: you can redistribute it and/or modify

 *  it under the terms of the GNU General Public License as published by

 *  the Free Software Foundation, either version 2 of the License, or

 *  any later version.

 *

 *  Eyepin For Wordpress is distributed in the hope that it will be useful,

 *  but WITHOUT ANY WARRANTY; without even the implied warranty of

 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

 *  GNU General Public License for more details.

 *

 *  You should have received a copy of the GNU General Public License

 *  along with Eyepin For Wordpress. If not, see http://www.gnu.org/copyleft/gpl.html.

 *

 *  This copyright notice MUST APPEAR in all copies of the script!

 * ************************************************************************************** */


defined('ABSPATH') or die('No script kiddies please!');


if (!defined('EYEPIN_CON_SUCCESS')) /**
 * Status code for succesful connection to eyepin server.
 */ {
    define('EYEPIN_CON_SUCCESS', 0);
}

if (!defined('EYEPIN_CON_ERR_CONNECTION')) /**
 * Error code for failed connection to eyepin server.
 */ {
    define('EYEPIN_CON_ERR_CONNECTION', 1);
}

if (!defined('EYEPIN_CON_ERR_EYEPIN')) /**
 * Error code for internal error of eyepin server.
 */ {
    define('EYEPIN_CON_ERR_EYEPIN', 2);
}


class VA_EY4WP_Api_Eyepin {


    /**
     * The host of the eyepin server.
     *
     * @var string
     * @access protected
     */

    var $host = "apiv3.eyepin.com";


    /**
     * The username of the eyepin account.
     *
     * @var string
     * @access protected
     */

    var $username;


    /**
     * The password of the eyepin account.
     *
     * @var string
     * @access protected
     */

    var $password;


    /**
     * The script name of the eyepin server interface.
     *
     * @var string
     * @access protected
     */

    var $interfacePath = "/interface3.php";


    /**
     * The response of the eyepin server to the sent command.
     *
     * @var string
     * @access protected
     */

    var $response;


    /**
     * The used character set for the xml string of several commands.
     *
     * @var string
     * @access protected
     */

    var $charset = 'utf-8';


    /**
     * List of valid fields for xml data sets used in several commands.
     *
     * @var array
     * @access protected
     */

    var $validFields = array(

        'address' => array(

            'email',

            'salutation',

            'title',

            'firstname',

            'lastname',

            'street',

            'zip',

            'city',

            'country',

            'phone',

            'fax',

            'mobile',

            'internet',

            'company',

            'function',

            'attribute1',

            'attribute2',

            'attribute3',

            'attribute4',

            'attribute5',

            'attribute6',

            'attribute7',

            'attribute8',

            'attribute9',

            'attribute10',

            'attribute11',

            'attribute12',

            'attribute13',

            'attribute14',

            'attribute15',

            'attribute16',

            'attribute17',

            'attribute18',

            'attribute19',

            'attribute20',

            'status',

            'language',

            'newsletterformat',

            'handleoptin',

            'handleoptout',

        ),

        'content' => array(

            'id',

            'title',

            'opener',

            'detail',

            'catid',

            'detailurl',

            'detailname',

            'pictureurl',

            'mediaitemid',

        ),

        'account' => array(

            'name',

            'type',

            'customers'

        ),

    );


    /**
     * Setting if the connection class should operate silently or if it
     * should generate user errors upon connection problems. The class
     * operates in silent mode for default.
     *
     * @var boolean
     * @access public
     */

    var $silent = TRUE;


    /**
     * Constructor of the class.
     *
     * @param string $host The host of the eyepin server.
     * @param string $username The username of the used eyepin account.
     * @param string $password The password of the used eyepin account.
     *
     * @access public
     */

    function __construct($host = NULL, $username = NULL, $password = NULL) {

        $this->setHost($host);

        $this->setUsername($username);

        $this->setPassword($password);

    }


    /**
     * Send command to eyepin server
     *
     * Implemented commands are: AddressInsert, AddressSignout, ContentInsert, ContentUpdate, ContentDelete
     *
     * The data has to be given in the form of an associative array.
     * For further documentation about valid field names for the different
     * commands and what those fields and commands mean we refer you to
     * the interface documentation of the eyepin eMarketing Software.
     *
     * @param string $command The command
     * @param array  $data    The data array for the command.
     *
     * @return integer  Status code of operation. See {@link class.tx_eyepin_connection.php}
     * @access public
     */

    function sendCommand($command, $data = array()) {


        if (isset($this->host) && isset($this->username) && isset($this->password)) {


            switch ($command) {

                case 'AddressInsert':

                    $xmlRequest = $this->createXml('addressinsert', 'address', $data);

                    return $this->sendRequest($command, $xmlRequest);

                case 'AddressDelete':

                    $xmlRequest = $this->createXml('addressdelete', 'address', $data);

                    return $this->sendRequest($command, $xmlRequest);

                case 'ContentInsert':

                    $xmlRequest = $this->createXml('contentinsert', 'content', $data);

                    return $this->sendRequest($command, $xmlRequest);

                case 'ContentUpdate':

                    $xmlRequest = $this->createXml('contentinsert', 'content', $data);

                    return $this->sendRequest($command, $xmlRequest);

                case 'ContentDelete':

                    $xmlRequest = $this->createXml('contentdelete', 'content', $data);

                    return $this->sendRequest($command, $xmlRequest);

                case 'GetAccountInfo':

                    $xmlRequest = $this->createXml('getaccountinfo', 'account', $data);

                    return $this->sendRequest($command, $xmlRequest);

                default:

                    if (!$this->silent) {

//                        trigger_error(
//                            "tx_eyepin_connection::sendCommand() - Unknown command has been passed!", E_USER_WARNING
//                        );

                    }

                    return EYEPIN_CON_ERR_CONNECTION;

            }

        } else {

            if (!$this->silent) {

//                trigger_error(
//                    "tx_eyepin_connection::sendCommand() - Host, username, or password not set!", E_USER_WARNING
//                );

            }

            return FALSE;

        }

    }

    /**
     * Sends a request to the eyepin server.
     *
     * This method actualy sends a command via a HTTP request to the eyepin
     * server. Please do not use this method directly, use instead {@link sendCommand()}.
     *
     * @param string $command The command to send to the eyepin server.
     * @param mixed  $data    The data to send to the eyepin server. This can be a XML string
     *                        or a integer value or whatever the given command needs as data.
     *
     * @return integer  Status code of operation. See {@link class.tx_eyepin_connection.php}
     * @access protected
     */

    function sendRequest($command, $data) {

        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode($this->username . ':' . $this->password)
            ),
            'body' => array(
                'data' => $data,
                'cmd' => $command
            ),
            'sslverify' => true
        );

        $response = wp_remote_post('https://' . $this->host, $args);

        if ( is_wp_error(  $response ) ) {
            wp_die("Error in response: " . esc_html($response->get_error_message()));
        } else {
            $this->response = wp_remote_retrieve_body($response);
        }

        if ($this->response != FALSE) {

            $xml = simplexml_load_string($this->response, 'SimpleXMLElement', LIBXML_NOCDATA);


            if ($xml->code == "2000") {

                if (!$this->silent) {
                    wp_die("Command sent successfuly.");
                }

                return EYEPIN_CON_SUCCESS;

            } else if ($xml->code == "4108"){

                if (!$this->silent) {
                    wp_die("Overwrite Contact is not enabled.");

                }


                return intval($xml->code);
            } else {

                if (!$this->silent) {
                    wp_die( "Eyepin server responded with error message: '"
                        . esc_html($xml->code) . "'");
                }

                return EYEPIN_CON_ERR_EYEPIN;
            }

        } else {

            if (!$this->silent) {
                wp_die("Connection to server failed!");
            }

            return EYEPIN_CON_ERR_CONNECTION;
        }
    }

    /**
     * Create XML structure based on the given root node name and the data array.
     *
     * This method creates a XML string with all values in the given data array
     * that are defined in the class member {@link $validFields} under the
     * given root node name.
     *
     * If the valid fields array is for example defined like the following:
     * <pre>
     * var $validFields = array (
     *        'address' => array (
     *                'email',
     *                'firstname',
     *                'lastname',
     *            }
     *        }
     * </pre>
     * Then the item <samp>$data['firstname'] = John;</samp> would become
     * <samp><firstname>John</firstname></samp> in the XML string, but an
     * item like <samp>$data['age'] = 80;</samp> would not be included in
     * the XML string.
     *
     * @param string $rootNodeName The name of the root node
     * @param array  $data         Associative data array for XML structure
     *
     * @return string  XML structure
     * @access protected
     */

    function createXml($rootNodeName, $validFieldRoot, $data) {

        $xml = '<?xml version="1.0" encoding="utf-8"?>';


        if ($rootNodeName == "contentinsert") {

            if (isset($data['eyepin_id'])) {

                $xml .= '<' . $rootNodeName . ' id="' . $data['eyepin_id'] . '" customerid="' . get_option(
                        VA_EY4WP_PLUGIN_NAME . '_eyepin_client', ''
                    ) . '">';


                unset($data['eyepin_id']);

            } else {

                $xml .= '<' . $rootNodeName . ' customerid="' . get_option(VA_EY4WP_PLUGIN_NAME . '_eyepin_client', '')
                    . '">';

            }

        } else {
            if ($rootNodeName == "contentdelete") {

                $xml .= '<' . $rootNodeName . ' customerid="' . get_option(VA_EY4WP_PLUGIN_NAME . '_eyepin_client', '')
                    . '" id="' . $data['id'] . '"/>';

                return $xml;

            } else {
                if ($rootNodeName == "addressinsert") {

                    // get setting 'overwrite_address' from DB for $updateAddress
                    $overwriteAddress = 0;

                    global $wpdb;
                    $sql = "SELECT * FROM {$wpdb->prefix}" . VA_EY4WP_Admin_Pages::TABLE_NAME;
//                    $result = $wpdb->get_results($sql, 'ARRAY_A');

                    $table_name = "{$wpdb->prefix}va_ey4wp_settings";
                    $result = $wpdb->get_results($wpdb->prepare("SELECT * FROM %i", $table_name), 'ARRAY_A'); // phpcs:ignore WordPress.DB.DirectDatabaseQuery

                    if ($result){
                        foreach ($result as $k => $v){
                            if ($v["name"] == "overwrite_address"){
                                $overwriteAddress = $v["value"];
                            }
                        }
                    }



                    $xml .= '<' . $rootNodeName . ' customerid="' . get_option(
                            VA_EY4WP_PLUGIN_NAME . '_eyepin_client', ''
                        ) . '" overrule-blacklist="1" save-empty-fields="0" key="Email" update="' . $overwriteAddress .'">';

                        $xml .= '<typesignin>SignIn</typesignin>';



                } else {
                    if ($rootNodeName == "addressdelete") {

                        $xml .= '<' . $rootNodeName . ' customerid="' . get_option(
                                VA_EY4WP_PLUGIN_NAME . '_eyepin_client', ''
                            ) . '" key="Email">';

                    } else {
                        if ($rootNodeName == "getaccountinfo") {

                            $xml .= '<' . $rootNodeName . '>';

                        } else {

                            $xml .= '<' . $rootNodeName . ' customerid="' . get_option(
                                    VA_EY4WP_PLUGIN_NAME . '_eyepin_client', ''
                                ) . '" overrule-blacklist="0" save-empty-fields="0" key="Email">';

                        }
                    }
                }
            }
        }


        if ($rootNodeName == "contentinsert" || $rootNodeName == "contentupdate" || $rootNodeName == "addressinsert"){
            $xml .= "<language>{$data['lang']}</language>";
        }




        foreach ($data as $field => $value) {

            if (in_array($field, $this->validFields[$validFieldRoot])) {

                $xmlValue = $value;

                if ($field == "title" || $field == "opener" || $field == "detail") {

                    $xmlValue = '<![CDATA[ ' . $value . ' ]]>';

                }

                $xml .= '<' . $field . '>' . utf8_decode($xmlValue) . '</' . $field . '>';

            }

        }

        $xml .= '</' . $rootNodeName . '>';

        //echo htmlentities( $xml);die();

        return $xml;

    }


    /**
     * Set the host name of the eyepin server.
     *
     * @param string $host
     *
     * @access public
     */

    function setHost($host) {
        if(!empty($host)) {
            if (substr($host, -1) == '/') {
                $host = substr($host, 0, -1);
            }

            if (substr($host, 0, 8) == 'https://') {
                $host = substr($host, 8);

            }

            $this->host = $host;
        }
    }


    /**
     * Returns the set host of the eyepin server.
     *
     * @return string
     * @access public
     */

    function getHost() {

        return $this->host;

    }


    /**
     * Set the username of the eyepin account.
     *
     * @param string $username
     *
     * @access public
     */

    function setUsername($username) {

        $this->username = $username;

    }


    /**
     * Returns the set username of the eyepin account
     *
     * @return string
     * @access public
     */

    function getUsername() {

        return $this->username;

    }


    /**
     * Set the password of the eyepin account
     *
     * @param string $password
     *
     * @access public
     */

    function setPassword($password) {

        $this->password = $password;

    }


    /**
     * Returns the set password of the eyepin account
     *
     * @return string
     * @access public
     */

    function getPassword() {

        return $this->password;

    }


    /**
     * Set the character set for XML strings in command requests.
     *
     * @param string $charset
     *
     * @access public
     */

    function setCharset($charset) {

        $this->charset = $charset;

    }


    /**
     * Returns the set character set for XML strings in command requests.
     *
     * @return string
     * @access public
     */

    function getCharset() {

        return $this->charset;

    }


    /**
     * Returns the last response from the eyepin server.
     *
     * @return string
     * @access public
     */

    function getResponse() {

        return simplexml_load_string($this->response, 'SimpleXMLElement', LIBXML_NOCDATA);

    }


}
